#!/usr/bin/env python3
"""
=========================================================================
Collects all standard modules with paths, and inserts them into the
config file for PyInstaller, so app users can run near-arbitrary Python
code without installing a Python separately.  Else, frozen apps/exes
include _only_ modules used by the frozen script.  Run me from a build.py
or other script and name this folder as a hooks dir in PyInstaller cmd.
Users can also give a Python and sys.path extensions in textConfig.py.

Additional docs clipped here: see ../widows/include-full-stdlib.py.
=========================================================================
"""

import os
import os.path
import sys
import pprint

# or via sys.executable?
path = '/usr/lib/python3.5'
#pyso = '/usr/lib/python3.5/lib-dynload'   # subdir: found on main walk here

traceonly = False
modules = []

skip = ['site-packages',    # skip local extensions (but see caveat above)
        'test',             # skip py selftest (and other), and idle dev gui 
        'idlelib']          # dropped lib-tk (why? + PyEdit adds tkinter anyhow)

skip += ['lib2to3',         # added: skip the 2.X=>3.X converter utility
         'tests',           # added: skip individual package tests
         'config-3.5m-x86_64-linux-gnu']   # fails build, seems pointless

print('Building full stdlib list')

# not for Linux: special-case dynamic-load mods (found by walk on Linux)

# walk the main Lib folder
for root, dirs, files in os.walk(path):
    # prune explicit-skip dirs
    for item in dirs.copy():
        if item in skip:
            print('Omitting:', os.path.join(root, item))
            dirs.remove(item)

    # not for Linux: prune plat-* folders: not on sys.path, unimportable

    # process files in this root
    for f in files:
        d = root.replace(path, '')
        full_path = os.path.join(d, f)

        # Fix up slashes
        if '\\' in full_path:
            full_path = full_path.replace('\\', '.')
        if '/' in full_path:
            full_path = full_path.replace('/', '.')
        if full_path[0] == '.':
            full_path = full_path[1:]

        # Collapse some auto-visible subdirs (on sys.path)
        if full_path.startswith('plat-x86_64-linux-gnu'):  # importable mods
            full_path = full_path[21:]
        if full_path.startswith('lib-dynload.'):           # linux .so dir
            full_path = full_path[12:]
        if full_path.startswith('.'):                      # what for, this?
            continue                                       # don't drop here
        
        # Ignore .pyc and .pyo files
        if f.endswith(('.py', '.so')) and not f.startswith('_'):

            # added unportable .so hack: fix me
            cmodprefix = '.cpython-35m-x86_64-linux-gnu.so'
            if f.endswith(cmodprefix):
                full_path = full_path[:-len(cmodprefix)] 
            else:
                full_path = full_path[:-3]

            # Save each part of the file path as part of the module name:
            # foo.foo1.foo2.py has a package foo, a sub-package foo1,
            # and a module foo2. Save foo, foo.foo1, and foo.foo1.foo2.

            section_total = full_path.count('.')
            start = 0
            for x in range(section_total):
                stop = full_path.find('.', start)
                if stop != -1:
                    package = full_path[:stop]
                    if package and package not in modules:
                        modules.append(package)
                start = stop + 1
        
            if full_path and full_path not in modules:
                modules.append(full_path)

print('The number of modules is', len(modules))
for mod in modules:
    print(mod)
if traceonly: sys.exit()

# Store in a PyInstaller hook file in this dir
hook_file = open('hook-os.py', 'w', encoding='utf8')              # added utf8
formatmodules = pprint.pformat(modules)                           # not one line
hook_file.write('# Generated by %s\n' % __file__)
hook_file.write("print('\\n**RUNNING PYEDIT HOOK FILE**\\n')\n")
hook_file.write('hiddenimports = ' + formatmodules)
hook_file.close()
