#!/usr/bin/python
"""
--------------------------------------------------------------------------------
iconify.py, version 1.0, October 2014, author Mark Lutz.
License: provided freely, but with no warranties of any kind.

Package all image files in a folder in a ".ico" Windows icon file.
For file format, try http://en.wikipedia.org/wiki/ICO_(file_format),
http://msdn.microsoft.com/en-us/library/ms997538.aspx, or a search.

Runs on Python 2.X or 3.X, requires Pillow (PIL) image library:
fetch and install Python from https://www.python.org/downloads/;
fetch and install Pillow from https://pypi.python.org/pypi/Pillow.

Usage (the new icon file is written in '.'):
    [py[thon]] iconify.py [imagedir, else asks] [iconname-no-.ico, else asks]

TBD: could also PIL.Image.resize((32, 32)) if needed to shrink.
TBD: could rely on PIL for some manual actions here (e.g., loads).
--------------------------------------------------------------------------------
"""

import sys, os, glob, struct
import PIL.Image
if sys.version[0] == '2':
    input = raw_input       # 2.X compatibility

#-------------------------------------------------------------------------------
# Get parameters, image files
#-------------------------------------------------------------------------------

imagedir = sys.argv[1] if len(sys.argv) >= 2 else input('imagedir?')
iconname = sys.argv[2] if len(sys.argv) >= 3 else input('iconname?')    # no '.ico'
PNG_SUPPORT, BMP_SUPPORT = True, False  # BMP is TBD

imagefiles = []
if PNG_SUPPORT:
    imagefiles += glob.glob(os.path.join(imagedir, '*.png'))
if BMP_SUPPORT:
    imagefiles += glob.glob(os.path.join(imagedir, '*.bmp'))

if len(imagefiles) == 0:
    print('No image files found!')
    input('Press Return to exit.')
    sys.exit(1)

#-------------------------------------------------------------------------------
# Load 1..N image files' contents (bytes)
#-------------------------------------------------------------------------------

imagedatas = []
for image in imagefiles:
    imagefile = open(image, 'rb')
    imagedata = imagefile.read()
    imagefile.close()
    imagedatas.append(imagedata)

#-------------------------------------------------------------------------------
# Package in 1 ".ico" file structure
#-------------------------------------------------------------------------------

iconfile = open(iconname + '.ico', 'wb')

# 1) HEADER: 3 2-byte ints, little-endian for all items

hdrsize = 6
hdrfmt  = "<hhh"
iconfile.write(struct.pack(hdrfmt,
                0,                       # reserved, always 0
                1,                       # 1=.ico icon, 2=.cur cursor 
                len(imagedatas)          # number images in file
                ))

# 2) DIRECTORY: 1..N 16-byte records, bytes/shorts/ints, for width/height/size/offset/etc.

dirsize = 16
dirfmt  = "<BBBBhhii"
imgoffset = hdrsize + (dirsize * len(imagedatas))

# bit per pixel: this table seems a hack, but no direct map in PIL itself?
# example: common PNG => 'RGBA' => 32bpp (8 bits (byte) * 4 bands (RGBA))
mode_to_bpp = {'1': 1, 'L': 8, 'P': 8,
               'RGB': 24, 'RGBA': 32, 'CMYK': 32, 'YCbCr': 24, 'I': 32, 'F': 32}

for (imagedata, imagefile) in zip(imagedatas, imagefiles):
    
    pilimg = PIL.Image.open(imagefile)
    width, height = pilimg.size          # a 2-tuple
    if width  >= 256: width  = 0         # 0 means 256 (or more), per spec (and practice)
    if height >= 256: height = 0
    bitsperpixel = mode_to_bpp[pilimg.mode]
   
    iconfile.write(struct.pack(dirfmt,
                width,                   # width, in pixels,  0..255
                height,                  # height, in pixels, 0..255
                0,                       # ?color count/palette (0 if >= 8bpp)
                0,                       # reserved, always 0
                1,                       # ?color planes: 0 or 1 treated same (>1 matters)
                bitsperpixel,            # ?bits per pixel (0 apparently means inferred)
                len(imagedata),          # size of image data in bytes
                imgoffset                # offset to image data from start of file            
                ))
    imgoffset += len(imagedata)

# 3) IMAGE BYTES: packed into rest of icon file sequentially

for imagedata in imagedatas:
    iconfile.write(imagedata)

iconfile.close()
print('Finished: see %s' % os.path.abspath(iconname + '.ico'))
if sys.platform.startswith('win') and sys.stdout.isatty():
    input('Press Return to close.')

