'use strict';

const requireUncached = require('./_utils/requireUncached');
const { describe, it, before, beforeEach } = require('node:test');
const assert = require('assert');

describe('Tests for strict mode', function() {
  describe("Specifying an unused NODE_ENV value and valid NODE_APP_INSTANCE value throws an exception", _expectException({
    NODE_ENV         : 'BOOM',
    APP_INSTANCE     : 'valid-instance',
    exceptionMessage : "FATAL: NODE_ENV value of 'BOOM' did not match any deployment config file names. "
                     + "See https://github.com/node-config/node-config/wiki/Strict-Mode",
  }));

  // Because NODE_ENV=development = default
  describe("Specifying NODE_ENV=development with no development file does not throw an exception", _expectException({
    NODE_ENV         : 'development',
    APP_INSTANCE     : 'valid-instance',
    exceptionMessage : undefined,
  }));

  describe("Specifying NODE_ENV=production,cloud with no cloud file throws an exception", _expectException({
    NODE_ENV         : 'production,cloud',
    APP_INSTANCE     : 'valid-instance',
    exceptionMessage : "FATAL: NODE_ENV value of 'cloud' did not match any deployment config file names. "
                     + "See https://github.com/node-config/node-config/wiki/Strict-Mode"
  }));

  describe("Specifying an unused NODE_APP_INSTANCE and valid NODE_ENV value throws an exception", _expectException({
    NODE_ENV         : 'valid-deployment',
    APP_INSTANCE     : 'BOOM',
    exceptionMessage : "FATAL: NODE_APP_INSTANCE value of 'BOOM' did not match any instance config file names. "
                     + "See https://github.com/node-config/node-config/wiki/Strict-Mode",
  }));

  describe("NODE_ENV=default throws exception: reserved word", _expectException({
    NODE_ENV         : 'default',
    APP_INSTANCE     : 'valid-instance',
    exceptionMessage :"FATAL: NODE_ENV value of 'default' is ambiguous. "
                     +"See https://github.com/node-config/node-config/wiki/Strict-Mode",
  }));

  describe("NODE_ENV=production,default throws exception: reserved word", _expectException({
    NODE_ENV         : 'production,default',
    APP_INSTANCE     : 'valid-instance',
    exceptionMessage :"FATAL: NODE_ENV value of 'default' is ambiguous. "
                     +"See https://github.com/node-config/node-config/wiki/Strict-Mode",
  }));

  describe("NODE_ENV=local throws exception: reserved word", _expectException({
    NODE_ENV         : 'local',
    APP_INSTANCE     : 'valid-instance',
    exceptionMessage :"FATAL: NODE_ENV value of 'local' is ambiguous. "
                     +"See https://github.com/node-config/node-config/wiki/Strict-Mode",
  }));

  describe("Specifying reserved word for NODE_CONFIG_ENV throws reserved word exception with appropriate wording", _expectException({
    NODE_CONFIG_ENV  : 'local',
    APP_INSTANCE     : 'valid-instance',
    exceptionMessage :"FATAL: NODE_CONFIG_ENV value of 'local' is ambiguous. "
                     +"See https://github.com/node-config/node-config/wiki/Strict-Mode",
  }));

  describe("Specifying NODE_CONFIG_ENV=production,cloud with no cloud file throws an exception with appropriate wording", _expectException({
    NODE_CONFIG_ENV  : 'cloud',
    APP_INSTANCE     : 'valid-instance',
    exceptionMessage :"FATAL: NODE_CONFIG_ENV value of 'cloud' did not match any deployment config file names. "
                     +"See https://github.com/node-config/node-config/wiki/Strict-Mode",
  }));
});


// helper function to create similar tests
// Pass in NODE_ENV and APP_INSTANCE and an 'exceptionMessage' that is' expected to be generated by the combo under strict mode.
function _expectException (opts) {
  let error;

  return () => {
    beforeEach(function() {
      // Change the configuration directory for testing
      process.env.NODE_CONFIG_DIR         = __dirname + '/6-config';
      process.env.NODE_CONFIG_STRICT_MODE = 1;
      process.env.NODE_APP_INSTANCE       = opts.APP_INSTANCE;

      if (!!opts.NODE_ENV) {
        process.env.NODE_ENV              = opts.NODE_ENV;
      }

      if (!!opts.NODE_CONFIG_ENV) {
        process.env.NODE_CONFIG_ENV       = opts.NODE_CONFIG_ENV;
      }

      delete process.env.NODE_CONFIG;
      try {
        let config = requireUncached(__dirname + '/../lib/config');
      }
      catch (e) {
        error = e;
      }
    });

    it('Exception is an error object', function() {
      // Allow case for exceptionMessage=null to indicate no error
      if (opts.exceptionMessage) {
        assert.ok(error instanceof Error);
      }
    });

    it('Exception contains expected string', function () {
      // This conditional allows to test for error===null
      if (error) {
        assert.strictEqual(error.message, opts.exceptionMessage );
      }
      else {
        assert.strictEqual(error, opts.exceptionMessage );
      }
    });
  };
}
